include <../std.scad>


module test_square() {
    assert(square(100, center=true) == [[50,-50],[-50,-50],[-50,50],[50,50]]);
    assert(square(100, center=false) == [[100,0],[0,0],[0,100],[100,100]]);
    assert(square(100, anchor=FWD+LEFT) == [[100,0],[0,0],[0,100],[100,100]]);
    assert(square(100, anchor=BACK+RIGHT) == [[0,-100],[-100,-100],[-100,0],[0,0]]);
}
test_square();


module test_circle() {
    for (pt = circle(d=200)) {
        assert(approx(norm(pt),100));
    }
    for (pt = circle(r=100)) {
        assert(approx(norm(pt),100));
    }
    assert(is_polygon_clockwise(circle(d=200)));
    assert(is_polygon_clockwise(circle(r=100)));
    assert(len(circle(d=100,$fn=6)) == 6);
    assert(len(circle(d=100,$fn=36)) == 36);
}
test_circle();



module test_rect() {
    assert_equal(rect(100,anchor=CENTER), [[50,-50],[-50,-50],[-50,50],[50,50]]);
    assert_equal(rect(100), [[50,-50],[-50,-50],[-50,50],[50,50]]);    
    assert_equal(rect([100,80],anchor=CENTER), [[50,-40],[-50,-40],[-50,40],[50,40]]);
    assert_equal(rect([100,80]), [[50,-40],[-50,-40],[-50,40],[50,40]]);
    assert_equal(rect([100,80],anchor=FRONT+LEFT), [[100,0],[0,0],[0,80],[100,80]]);
    assert_equal(rect([100,80],anchor=BACK+RIGHT), [[0,-80],[-100,-80],[-100,0],[0,0]]);
    assert_approx(rect([100,80],rounding=10,anchor=CENTER,$fn=12), [[50,-30],[48.6602540378,-35],[45,-38.6602540378],[40,-40],[-40,-40],[-45,-38.6602540378],[-48.6602540378,-35],[-50,-30],[-50,30],[-48.6602540378,35],[-45,38.6602540378],[-40,40],[40,40],[45,38.6602540378],[48.6602540378,35],[50,30]]);
}
test_rect();


module test_trapezoid() {
    assert_equal(trapezoid(w1=100,w2=80,h=60,anchor=CENTER), [[50,-30],[-50,-30],[-40,30],[40,30]]);
}
test_trapezoid();


module test_ellipse() {
    assert_approx(ellipse(d=100,$fn=24), [[50,0],[48.2962913145,-12.9409522551],[43.3012701892,-25],[35.3553390593,-35.3553390593],[25,-43.3012701892],[12.9409522551,-48.2962913145],[0,-50],[-12.9409522551,-48.2962913145],[-25,-43.3012701892],[-35.3553390593,-35.3553390593],[-43.3012701892,-25],[-48.2962913145,-12.9409522551],[-50,0],[-48.2962913145,12.9409522551],[-43.3012701892,25],[-35.3553390593,35.3553390593],[-25,43.3012701892],[-12.9409522551,48.2962913145],[0,50],[12.9409522551,48.2962913145],[25,43.3012701892],[35.3553390593,35.3553390593],[43.3012701892,25],[48.2962913145,12.9409522551]]);
    assert_approx(ellipse(d=[100,80],$fn=24), [[50,0],[48.2962913145,-10.3527618041],[43.3012701892,-20],[35.3553390593,-28.2842712475],[25,-34.6410161514],[12.9409522551,-38.6370330516],[0,-40],[-12.9409522551,-38.6370330516],[-25,-34.6410161514],[-35.3553390593,-28.2842712475],[-43.3012701892,-20],[-48.2962913145,-10.3527618041],[-50,0],[-48.2962913145,10.3527618041],[-43.3012701892,20],[-35.3553390593,28.2842712475],[-25,34.6410161514],[-12.9409522551,38.6370330516],[0,40],[12.9409522551,38.6370330516],[25,34.6410161514],[35.3553390593,28.2842712475],[43.3012701892,20],[48.2962913145,10.3527618041]]);
}
test_ellipse();


module test_star() {
    assert_approx(star(n=5,d=100,step=2), [[50,0],[15.4508497187,-11.2256994145],[15.4508497187,-47.5528258148],[-5.90169943749,-18.1635632001],[-40.4508497187,-29.3892626146],[-19.0983005625,0],[-40.4508497187,29.3892626146],[-5.90169943749,18.1635632001],[15.4508497187,47.5528258148],[15.4508497187,11.2256994145]]);
    assert_approx(star(n=6,d=100,step=2), [[50,0],[25,-14.4337567297],[25,-43.3012701892],[0,-28.8675134595],[-25,-43.3012701892],[-25,-14.4337567297],[-50,0],[-25,14.4337567297],[-25,43.3012701892],[0,28.8675134595],[25,43.3012701892],[25,14.4337567297]]);
    assert_approx(star(n=7,d=100,step=3), [[50,0],[16.0776033027,-7.74256568183],[31.1744900929,-39.0915741234],[3.97084009243,-17.3973871675],[-11.1260466978,-48.7463956091],[-11.1260466978,-13.951621274],[-45.0484433951,-21.6941869559],[-17.8447933946,0],[-45.0484433951,21.6941869559],[-11.1260466978,13.951621274],[-11.1260466978,48.7463956091],[3.97084009243,17.3973871675],[31.1744900929,39.0915741234],[16.0776033027,7.74256568183]]);
}
test_star();


module test_regular_ngon() {
    assert_approx(regular_ngon(n=7,d=100), [[50,0],[31.1744900929,-39.0915741234],[-11.1260466978,-48.7463956091],[-45.0484433951,-21.6941869559],[-45.0484433951,21.6941869559],[-11.1260466978,48.7463956091],[31.1744900929,39.0915741234]]);
    assert_approx(regular_ngon(n=9,d=100), [[50,0],[38.3022221559,-32.1393804843],[8.68240888335,-49.2403876506],[-25,-43.3012701892],[-46.9846310393,-17.1010071663],[-46.9846310393,17.1010071663],[-25,43.3012701892],[8.68240888335,49.2403876506],[38.3022221559,32.1393804843]]);
    assert_approx(regular_ngon(n=5,d=100,rounding=10,$fn=24), [[47.4207962323,2.07911690818],[47.4207962323,-2.07911690818],[45.7294901688,-5.87785252292],[19.72135955,-41.6749732918],[16.6311896062,-44.4573748068],[12.6764742389,-45.7423397226],[8.5410196625,-45.3076859319],[-33.5410196625,-31.6344024975],[-37.1421557823,-29.5552855893],[-39.5863042952,-26.1912037653],[-40.4508497187,-22.1238373346],[-40.4508497187,22.1238373346],[-39.5863042952,26.1912037653],[-37.1421557823,29.5552855893],[-33.5410196625,31.6344024975],[8.5410196625,45.3076859319],[12.6764742389,45.7423397226],[16.6311896062,44.4573748068],[19.72135955,41.6749732918],[45.7294901688,5.87785252292]]);
}
test_regular_ngon();


module test_pentagon() {
    assert_approx(pentagon(d=100), [[50,0],[15.4508497187,-47.5528258148],[-40.4508497187,-29.3892626146],[-40.4508497187,29.3892626146],[15.4508497187,47.5528258148]]);
    assert_approx(pentagon(d=100,rounding=10,$fn=24), [[47.4207962323,2.07911690818],[47.4207962323,-2.07911690818],[45.7294901688,-5.87785252292],[19.72135955,-41.6749732918],[16.6311896062,-44.4573748068],[12.6764742389,-45.7423397226],[8.5410196625,-45.3076859319],[-33.5410196625,-31.6344024975],[-37.1421557823,-29.5552855893],[-39.5863042952,-26.1912037653],[-40.4508497187,-22.1238373346],[-40.4508497187,22.1238373346],[-39.5863042952,26.1912037653],[-37.1421557823,29.5552855893],[-33.5410196625,31.6344024975],[8.5410196625,45.3076859319],[12.6764742389,45.7423397226],[16.6311896062,44.4573748068],[19.72135955,41.6749732918],[45.7294901688,5.87785252292]]);
}
test_pentagon();


module test_hexagon() {
    assert_approx(hexagon(d=100), [[50,0],[25,-43.3012701892],[-25,-43.3012701892],[-50,0],[-25,43.3012701892],[25,43.3012701892]]);
    assert_approx(hexagon(d=100,rounding=10,$fn=24), [[48.3010721463,1.73648177667],[48.3010721463,-1.73648177667],[47.1132486541,-5],[27.8867513459,-38.3012701892],[25.654373405,-40.9617146204],[22.6466987414,-42.6981963971],[19.2264973081,-43.3012701892],[-19.2264973081,-43.3012701892],[-22.6466987414,-42.6981963971],[-25.654373405,-40.9617146204],[-27.8867513459,-38.3012701892],[-47.1132486541,-5],[-48.3010721463,-1.73648177667],[-48.3010721463,1.73648177667],[-47.1132486541,5],[-27.8867513459,38.3012701892],[-25.654373405,40.9617146204],[-22.6466987414,42.6981963971],[-19.2264973081,43.3012701892],[19.2264973081,43.3012701892],[22.6466987414,42.6981963971],[25.654373405,40.9617146204],[27.8867513459,38.3012701892],[47.1132486541,5]]);
}
test_hexagon();


module test_octagon() {
    assert_approx(octagon(d=100), [[50,0],[35.3553390593,-35.3553390593],[0,-50],[-35.3553390593,-35.3553390593],[-50,0],[-35.3553390593,35.3553390593],[0,50],[35.3553390593,35.3553390593]]);
    assert_approx(octagon(d=100,rounding=10,$fn=24), [[49.1760779971,0],[48.4148733222,-3.82683432365],[36.9404657371,-31.5285047357],[34.7727382239,-34.7727382239],[31.5285047357,-36.9404657371],[3.82683432365,-48.4148733222],[0,-49.1760779971],[-3.82683432365,-48.4148733222],[-31.5285047357,-36.9404657371],[-34.7727382239,-34.7727382239],[-36.9404657371,-31.5285047357],[-48.4148733222,-3.82683432365],[-49.1760779971,0],[-48.4148733222,3.82683432365],[-36.9404657371,31.5285047357],[-34.7727382239,34.7727382239],[-31.5285047357,36.9404657371],[-3.82683432365,48.4148733222],[0,49.1760779971],[3.82683432365,48.4148733222],[31.5285047357,36.9404657371],[34.7727382239,34.7727382239],[36.9404657371,31.5285047357],[48.4148733222,3.82683432365]]);
}
test_octagon();


module test_teardrop2d() {
    $fn=24;
    assert_approx(teardrop2d(r=50), [[0,70.7106781187],[35.3553390593,35.3553390593],[43.3012701892,25],[48.2962913145,12.9409522551],[50,0],[48.2962913145,-12.9409522551],[43.3012701892,-25],[35.3553390593,-35.3553390593],[25,-43.3012701892],[12.9409522551,-48.2962913145],[0,-50],[-12.9409522551,-48.2962913145],[-25,-43.3012701892],[-35.3553390593,-35.3553390593],[-43.3012701892,-25],[-48.2962913145,-12.9409522551],[-50,0],[-48.2962913145,12.9409522551],[-43.3012701892,25],[-35.3553390593,35.3553390593]]);
    assert_approx(teardrop2d(d=100), [[0,70.7106781187],[35.3553390593,35.3553390593],[43.3012701892,25],[48.2962913145,12.9409522551],[50,0],[48.2962913145,-12.9409522551],[43.3012701892,-25],[35.3553390593,-35.3553390593],[25,-43.3012701892],[12.9409522551,-48.2962913145],[0,-50],[-12.9409522551,-48.2962913145],[-25,-43.3012701892],[-35.3553390593,-35.3553390593],[-43.3012701892,-25],[-48.2962913145,-12.9409522551],[-50,0],[-48.2962913145,12.9409522551],[-43.3012701892,25],[-35.3553390593,35.3553390593]]);
    assert_approx(teardrop2d(r=50,cap_h=50), [[20.7106781187,50],[35.3553390593,35.3553390593],[43.3012701892,25],[48.2962913145,12.9409522551],[50,0],[48.2962913145,-12.9409522551],[43.3012701892,-25],[35.3553390593,-35.3553390593],[25,-43.3012701892],[12.9409522551,-48.2962913145],[0,-50],[-12.9409522551,-48.2962913145],[-25,-43.3012701892],[-35.3553390593,-35.3553390593],[-43.3012701892,-25],[-48.2962913145,-12.9409522551],[-50,0],[-48.2962913145,12.9409522551],[-43.3012701892,25],[-35.3553390593,35.3553390593],[-20.7106781187,50]]);
    assert_approx(teardrop2d(r=50,cap_h=50,ang=30), [[28.8675134595,50],[43.3012701892,25],[48.2962913145,12.9409522551],[50,0],[48.2962913145,-12.9409522551],[43.3012701892,-25],[35.3553390593,-35.3553390593],[25,-43.3012701892],[12.9409522551,-48.2962913145],[0,-50],[-12.9409522551,-48.2962913145],[-25,-43.3012701892],[-35.3553390593,-35.3553390593],[-43.3012701892,-25],[-48.2962913145,-12.9409522551],[-50,0],[-48.2962913145,12.9409522551],[-43.3012701892,25],[-28.8675134595,50]]);
}
test_teardrop2d();


module test_glued_circles() {
    $fn=24;
    assert_approx(glued_circles(r=15, spread=40, tangent=30), deduplicate([[35,0],[34.4888873943,-3.88228567654],[32.9903810568,-7.5],[30.6066017178,-10.6066017178],[27.5,-12.9903810568],[23.8822856765,-14.4888873943],[20,-15],[16.1177143235,-14.4888873943],[12.5,-12.9903810568],[12.5,-12.9903810568],[6.47047612756,-10.4928704942],[0,-9.64101615138],[-6.47047612756,-10.4928704942],[-12.5,-12.9903810568],[-12.5,-12.9903810568],[-16.1177143235,-14.4888873943],[-20,-15],[-23.8822856765,-14.4888873943],[-27.5,-12.9903810568],[-30.6066017178,-10.6066017178],[-32.9903810568,-7.5],[-34.4888873943,-3.88228567654],[-35,0],[-34.4888873943,3.88228567654],[-32.9903810568,7.5],[-30.6066017178,10.6066017178],[-27.5,12.9903810568],[-23.8822856765,14.4888873943],[-20,15],[-16.1177143235,14.4888873943],[-12.5,12.9903810568],[-6.47047612756,10.4928704942],[0,9.64101615138],[6.47047612756,10.4928704942],[12.5,12.9903810568],[12.5,12.9903810568],[16.1177143235,14.4888873943],[20,15],[23.8822856765,14.4888873943],[27.5,12.9903810568],[30.6066017178,10.6066017178],[32.9903810568,7.5],[34.4888873943,3.88228567654]]));
    assert_approx(glued_circles(d=30, spread=40, tangent=30),deduplicate( [[35,0],[34.4888873943,-3.88228567654],[32.9903810568,-7.5],[30.6066017178,-10.6066017178],[27.5,-12.9903810568],[23.8822856765,-14.4888873943],[20,-15],[16.1177143235,-14.4888873943],[12.5,-12.9903810568],[12.5,-12.9903810568],[6.47047612756,-10.4928704942],[0,-9.64101615138],[-6.47047612756,-10.4928704942],[-12.5,-12.9903810568],[-12.5,-12.9903810568],[-16.1177143235,-14.4888873943],[-20,-15],[-23.8822856765,-14.4888873943],[-27.5,-12.9903810568],[-30.6066017178,-10.6066017178],[-32.9903810568,-7.5],[-34.4888873943,-3.88228567654],[-35,0],[-34.4888873943,3.88228567654],[-32.9903810568,7.5],[-30.6066017178,10.6066017178],[-27.5,12.9903810568],[-23.8822856765,14.4888873943],[-20,15],[-16.1177143235,14.4888873943],[-12.5,12.9903810568],[-6.47047612756,10.4928704942],[0,9.64101615138],[6.47047612756,10.4928704942],[12.5,12.9903810568],[12.5,12.9903810568],[16.1177143235,14.4888873943],[20,15],[23.8822856765,14.4888873943],[27.5,12.9903810568],[30.6066017178,10.6066017178],[32.9903810568,7.5],[34.4888873943,3.88228567654]]));
    assert_approx(glued_circles(d=30, spread=30, tangent=45),deduplicate( [[30,0],[29.4888873943,-3.88228567654],[27.9903810568,-7.5],[25.6066017178,-10.6066017178],[22.5,-12.9903810568],[18.8822856765,-14.4888873943],[15,-15],[11.1177143235,-14.4888873943],[7.5,-12.9903810568],[4.3933982822,-10.6066017178],[4.3933982822,-10.6066017178],[3.1066017178,-9.61920798589],[1.60809538023,-8.99850633757],[0,-8.7867965644],[-1.60809538023,-8.99850633757],[-3.1066017178,-9.61920798589],[-4.3933982822,-10.6066017178],[-4.3933982822,-10.6066017178],[-7.5,-12.9903810568],[-11.1177143235,-14.4888873943],[-15,-15],[-18.8822856765,-14.4888873943],[-22.5,-12.9903810568],[-25.6066017178,-10.6066017178],[-27.9903810568,-7.5],[-29.4888873943,-3.88228567654],[-30,0],[-29.4888873943,3.88228567654],[-27.9903810568,7.5],[-25.6066017178,10.6066017178],[-22.5,12.9903810568],[-18.8822856765,14.4888873943],[-15,15],[-11.1177143235,14.4888873943],[-7.5,12.9903810568],[-4.3933982822,10.6066017178],[-3.1066017178,9.61920798589],[-1.60809538023,8.99850633757],[0,8.7867965644],[1.60809538023,8.99850633757],[3.1066017178,9.61920798589],[4.3933982822,10.6066017178],[4.3933982822,10.6066017178],[7.5,12.9903810568],[11.1177143235,14.4888873943],[15,15],[18.8822856765,14.4888873943],[22.5,12.9903810568],[25.6066017178,10.6066017178],[27.9903810568,7.5],[29.4888873943,3.88228567654]]));
    assert_approx(glued_circles(d=30, spread=30, tangent=-30), deduplicate([[30,0],[29.4888873943,-3.88228567654],[27.9903810568,-7.5],[25.6066017178,-10.6066017178],[22.5,-12.9903810568],[22.5,-12.9903810568],[11.6468570296,-17.4859000695],[0,-19.0192378865],[-11.6468570296,-17.4859000695],[-22.5,-12.9903810568],[-22.5,-12.9903810568],[-25.6066017178,-10.6066017178],[-27.9903810568,-7.5],[-29.4888873943,-3.88228567654],[-30,0],[-29.4888873943,3.88228567654],[-27.9903810568,7.5],[-25.6066017178,10.6066017178],[-22.5,12.9903810568],[-11.6468570296,17.4859000695],[0,19.0192378865],[11.6468570296,17.4859000695],[22.5,12.9903810568],[22.5,12.9903810568],[25.6066017178,10.6066017178],[27.9903810568,7.5],[29.4888873943,3.88228567654]]));
    assert_approx(glued_circles(d=30, spread=50, tangent=0),[[40, 0], [39.4888873943, -3.88228567654], [37.9903810568, -7.5], [35.6066017178, -10.6066017178], [32.5, -12.9903810568], [28.8822856765, -14.4888873943], [25, -15], [-25, -15], [-28.8822856765, -14.4888873943], [-32.5, -12.9903810568], [-35.6066017178, -10.6066017178], [-37.9903810568, -7.5], [-39.4888873943, -3.88228567654], [-40, 0], [-39.4888873943, 3.88228567654], [-37.9903810568, 7.5], [-35.6066017178, 10.6066017178], [-32.5, 12.9903810568], [-28.8822856765, 14.4888873943], [-25, 15], [25, 15], [28.8822856765, 14.4888873943], [32.5, 12.9903810568], [35.6066017178, 10.6066017178], [37.9903810568, 7.5], [39.4888873943, 3.88228567654]]);
}
test_glued_circles();


module test_supershape() {
    $fn = 24;
    assert_approx(supershape(step=0.5,m1=16,m2=16,n1=0.5,n2=0.5,n3=16,r=50), [[31.8072679929,0],[31.8254440703,-0.277736442783],[31.8800818691,-0.556468898942],[31.9715117231,-0.837203498428],[32.1002896772,-1.12096681644],[32.2672058831,-1.40881663388],[32.4732966814,-1.70185336507],[32.7198607165,-2.00123241238],[33.0084795316,-2.30817774075],[33.3410431535,-2.6239970035],[33.7197810671,-2.95009857994],[34.1472983973,-3.2880108612],[34.6266155102,-3.63940394304],[35.1612056414,-4.00611335337],[35.7550182057,-4.39016421046],[36.4124633674,-4.79379174563],[37.138314422,-5.21944970831],[37.9374570742,-5.66979094838],[38.8143785616,-6.14759362503],[39.7722470564,-6.65559159145],[40.8113899535,-7.19614916902],[41.9269541846,-7.77070164547],[43.1055530065,-8.37887071896],[44.3208250346,-9.01717376565],[45.5281335086,-9.67730351785],[46.6592118851,-10.344098238],[47.6184628391,-10.9935883798],[48.28369535,-11.5918896634],[48.5147863138,-12.09609478],[48.1730192886,-12.4583898795],[47.1505372707,-12.6339483844],[45.4034911139,-12.5915024719],[42.9767802608,-12.3233934346],[40.0076792879,-11.8508145072],[36.7032455714,-11.2213082803],[33.2989492854,-10.4991183806],[30.0149182764,-9.75243812881],[27.0253656239,-9.04256084558],[24.4481119546,-8.41816003875],[22.3518922366,-7.9152201722],[20.7749251664,-7.56145437968],[19.7499695415,-7.38421103185],[19.3382230386,-7.42324832601],[19.694017676,-7.75767986951],[21.2841774283,-8.59936587637],[29.3860838838,-12.1721144897],[21.1308561166,-8.96951626646],[19.4112814895,-8.44026540548],[18.9232178765,-8.42515941694],[19.1867630853,-8.74391169667],[20.0368461313,-9.3433347963],[21.4020804312,-10.2082687145],[23.2399637988,-11.3348877018],[25.5038753899,-12.7157632035],[28.119767384,-14.327737116],[30.9699106495,-16.1219150427],[33.8877770139,-18.0184506563],[36.6694926246,-19.9099100234],[39.1031278208,-21.6752176912],[41.0086532394,-23.201579673],[42.2740152165,-24.406914065],[42.8728805757,-25.2540566427],[42.8583650351,-25.7519037455],[42.3384321905,-25.945024615],[41.4449788762,-25.8976970902],[40.3074271381,-25.6786631194],[39.0361388798,-25.3503649976],[37.7156606466,-24.9634512129],[36.4049851418,-24.5554725336],[35.1415494465,-24.1521177904],[33.9464564616,-23.7695647099],[32.8294395437,-23.4170116495],[31.7929154287,-23.0989051482],[30.8349807857,-22.8166855306],[29.9514622525,-22.5700456868],[29.1372224177,-22.3577771159],[28.3869307184,-22.1783009509],[27.6954768621,-22.0299770253],[27.0581618444,-21.9112674292],[26.4707610325,-21.8208114794],[25.9295205637,-21.7574511416],[25.4311237807,-21.7202316307],[24.9726478461,-21.7083915808],[24.5515204017,-21.7213503826],[24.1654803459,-21.7586962359],[23.8125438851,-21.8201762946],[23.4909757462,-21.9056892714],[23.1992650151,-22.0152804732],[22.9361050064,-22.1491391425],[22.7003766384,-22.3075979943],[22.4911348888,-22.4911348888],[22.3075979943,-22.7003766384],[22.1491391425,-22.9361050064],[22.0152804732,-23.1992650151],[21.9056892714,-23.4909757462],[21.8201762946,-23.8125438851],[21.7586962359,-24.1654803459],[21.7213503826,-24.5515204017],[21.7083915808,-24.9726478461],[21.7202316307,-25.4311237807],[21.7574511416,-25.9295205637],[21.8208114794,-26.4707610325],[21.9112674292,-27.0581618444],[22.0299770253,-27.6954768621],[22.1783009509,-28.3869307184],[22.3577771159,-29.1372224177],[22.5700456868,-29.9514622525],[22.8166855306,-30.8349807857],[23.0989051482,-31.7929154287],[23.4170116495,-32.8294395437],[23.7695647099,-33.9464564616],[24.1521177904,-35.1415494465],[24.5554725336,-36.4049851418],[24.9634512129,-37.7156606466],[25.3503649976,-39.0361388798],[25.6786631194,-40.3074271381],[25.8976970902,-41.4449788762],[25.945024615,-42.3384321905],[25.7519037455,-42.8583650351],[25.2540566427,-42.8728805757],[24.406914065,-42.2740152165],[23.201579673,-41.0086532394],[21.6752176912,-39.1031278208],[19.9099100234,-36.6694926246],[18.0184506563,-33.8877770139],[16.1219150427,-30.9699106495],[14.327737116,-28.119767384],[12.7157632035,-25.5038753899],[11.3348877018,-23.2399637988],[10.2082687145,-21.4020804312],[9.3433347963,-20.0368461313],[8.74391169667,-19.1867630853],[8.42515941694,-18.9232178765],[8.44026540548,-19.4112814895],[8.96951626646,-21.1308561166],[12.1721144897,-29.3860838838],[8.59936587637,-21.2841774283],[7.75767986951,-19.694017676],[7.42324832601,-19.3382230386],[7.38421103185,-19.7499695415],[7.56145437968,-20.7749251664],[7.9152201722,-22.3518922366],[8.41816003875,-24.4481119546],[9.04256084558,-27.0253656239],[9.75243812881,-30.0149182764],[10.4991183806,-33.2989492854],[11.2213082803,-36.7032455714],[11.8508145072,-40.0076792879],[12.3233934346,-42.9767802608],[12.5915024719,-45.4034911139],[12.6339483844,-47.1505372707],[12.4583898795,-48.1730192886],[12.09609478,-48.5147863138],[11.5918896634,-48.28369535],[10.9935883798,-47.6184628391],[10.344098238,-46.6592118851],[9.67730351785,-45.5281335086],[9.01717376565,-44.3208250346],[8.37887071896,-43.1055530065],[7.77070164547,-41.9269541846],[7.19614916902,-40.8113899535],[6.65559159145,-39.7722470564],[6.14759362503,-38.8143785616],[5.66979094838,-37.9374570742],[5.21944970831,-37.138314422],[4.79379174563,-36.4124633674],[4.39016421046,-35.7550182057],[4.00611335337,-35.1612056414],[3.63940394304,-34.6266155102],[3.2880108612,-34.1472983973],[2.95009857994,-33.7197810671],[2.6239970035,-33.3410431535],[2.30817774075,-33.0084795316],[2.00123241238,-32.7198607165],[1.70185336507,-32.4732966814],[1.40881663388,-32.2672058831],[1.12096681644,-32.1002896772],[0.837203498428,-31.9715117231],[0.556468898942,-31.8800818691],[0.277736442783,-31.8254440703],[0,-31.8072679929],[-0.277736442783,-31.8254440703],[-0.556468898942,-31.8800818691],[-0.837203498428,-31.9715117231],[-1.12096681644,-32.1002896772],[-1.40881663388,-32.2672058831],[-1.70185336507,-32.4732966814],[-2.00123241238,-32.7198607165],[-2.30817774075,-33.0084795316],[-2.6239970035,-33.3410431535],[-2.95009857994,-33.7197810671],[-3.2880108612,-34.1472983973],[-3.63940394304,-34.6266155102],[-4.00611335337,-35.1612056414],[-4.39016421046,-35.7550182057],[-4.79379174563,-36.4124633674],[-5.21944970831,-37.138314422],[-5.66979094838,-37.9374570742],[-6.14759362503,-38.8143785616],[-6.65559159145,-39.7722470564],[-7.19614916902,-40.8113899535],[-7.77070164547,-41.9269541846],[-8.37887071896,-43.1055530065],[-9.01717376565,-44.3208250346],[-9.67730351785,-45.5281335086],[-10.344098238,-46.6592118851],[-10.9935883798,-47.6184628391],[-11.5918896634,-48.28369535],[-12.09609478,-48.5147863138],[-12.4583898795,-48.1730192886],[-12.6339483844,-47.1505372707],[-12.5915024719,-45.4034911139],[-12.3233934346,-42.9767802608],[-11.8508145072,-40.0076792879],[-11.2213082803,-36.7032455714],[-10.4991183806,-33.2989492854],[-9.75243812881,-30.0149182764],[-9.04256084558,-27.0253656239],[-8.41816003875,-24.4481119546],[-7.9152201722,-22.3518922366],[-7.56145437968,-20.7749251664],[-7.38421103185,-19.7499695415],[-7.42324832601,-19.3382230386],[-7.75767986951,-19.694017676],[-8.59936587637,-21.2841774283],[-12.1721144897,-29.3860838838],[-8.96951626646,-21.1308561166],[-8.44026540548,-19.4112814895],[-8.42515941694,-18.9232178765],[-8.74391169667,-19.1867630853],[-9.3433347963,-20.0368461313],[-10.2082687145,-21.4020804312],[-11.3348877018,-23.2399637988],[-12.7157632035,-25.5038753899],[-14.327737116,-28.119767384],[-16.1219150427,-30.9699106495],[-18.0184506563,-33.8877770139],[-19.9099100234,-36.6694926246],[-21.6752176912,-39.1031278208],[-23.201579673,-41.0086532394],[-24.406914065,-42.2740152165],[-25.2540566427,-42.8728805757],[-25.7519037455,-42.8583650351],[-25.945024615,-42.3384321905],[-25.8976970902,-41.4449788762],[-25.6786631194,-40.3074271381],[-25.3503649976,-39.0361388798],[-24.9634512129,-37.7156606466],[-24.5554725336,-36.4049851418],[-24.1521177904,-35.1415494465],[-23.7695647099,-33.9464564616],[-23.4170116495,-32.8294395437],[-23.0989051482,-31.7929154287],[-22.8166855306,-30.8349807857],[-22.5700456868,-29.9514622525],[-22.3577771159,-29.1372224177],[-22.1783009509,-28.3869307184],[-22.0299770253,-27.6954768621],[-21.9112674292,-27.0581618444],[-21.8208114794,-26.4707610325],[-21.7574511416,-25.9295205637],[-21.7202316307,-25.4311237807],[-21.7083915808,-24.9726478461],[-21.7213503826,-24.5515204017],[-21.7586962359,-24.1654803459],[-21.8201762946,-23.8125438851],[-21.9056892714,-23.4909757462],[-22.0152804732,-23.1992650151],[-22.1491391425,-22.9361050064],[-22.3075979943,-22.7003766384],[-22.4911348888,-22.4911348888],[-22.7003766384,-22.3075979943],[-22.9361050064,-22.1491391425],[-23.1992650151,-22.0152804732],[-23.4909757462,-21.9056892714],[-23.8125438851,-21.8201762946],[-24.1654803459,-21.7586962359],[-24.5515204017,-21.7213503826],[-24.9726478461,-21.7083915808],[-25.4311237807,-21.7202316307],[-25.9295205637,-21.7574511416],[-26.4707610325,-21.8208114794],[-27.0581618444,-21.9112674292],[-27.6954768621,-22.0299770253],[-28.3869307184,-22.1783009509],[-29.1372224177,-22.3577771159],[-29.9514622525,-22.5700456868],[-30.8349807857,-22.8166855306],[-31.7929154287,-23.0989051482],[-32.8294395437,-23.4170116495],[-33.9464564616,-23.7695647099],[-35.1415494465,-24.1521177904],[-36.4049851418,-24.5554725336],[-37.7156606466,-24.9634512129],[-39.0361388798,-25.3503649976],[-40.3074271381,-25.6786631194],[-41.4449788762,-25.8976970902],[-42.3384321905,-25.945024615],[-42.8583650351,-25.7519037455],[-42.8728805757,-25.2540566427],[-42.2740152165,-24.406914065],[-41.0086532394,-23.201579673],[-39.1031278208,-21.6752176912],[-36.6694926246,-19.9099100234],[-33.8877770139,-18.0184506563],[-30.9699106495,-16.1219150427],[-28.119767384,-14.327737116],[-25.5038753899,-12.7157632035],[-23.2399637988,-11.3348877018],[-21.4020804312,-10.2082687145],[-20.0368461313,-9.3433347963],[-19.1867630853,-8.74391169667],[-18.9232178765,-8.42515941694],[-19.4112814895,-8.44026540548],[-21.1308561166,-8.96951626646],[-29.3860838838,-12.1721144897],[-21.2841774283,-8.59936587637],[-19.694017676,-7.75767986951],[-19.3382230386,-7.42324832601],[-19.7499695415,-7.38421103185],[-20.7749251664,-7.56145437968],[-22.3518922366,-7.9152201722],[-24.4481119546,-8.41816003875],[-27.0253656239,-9.04256084558],[-30.0149182764,-9.75243812881],[-33.2989492854,-10.4991183806],[-36.7032455714,-11.2213082803],[-40.0076792879,-11.8508145072],[-42.9767802608,-12.3233934346],[-45.4034911139,-12.5915024719],[-47.1505372707,-12.6339483844],[-48.1730192886,-12.4583898795],[-48.5147863138,-12.09609478],[-48.28369535,-11.5918896634],[-47.6184628391,-10.9935883798],[-46.6592118851,-10.344098238],[-45.5281335086,-9.67730351785],[-44.3208250346,-9.01717376565],[-43.1055530065,-8.37887071896],[-41.9269541846,-7.77070164547],[-40.8113899535,-7.19614916902],[-39.7722470564,-6.65559159145],[-38.8143785616,-6.14759362503],[-37.9374570742,-5.66979094838],[-37.138314422,-5.21944970831],[-36.4124633674,-4.79379174563],[-35.7550182057,-4.39016421046],[-35.1612056414,-4.00611335337],[-34.6266155102,-3.63940394304],[-34.1472983973,-3.2880108612],[-33.7197810671,-2.95009857994],[-33.3410431535,-2.6239970035],[-33.0084795316,-2.30817774075],[-32.7198607165,-2.00123241238],[-32.4732966814,-1.70185336507],[-32.2672058831,-1.40881663388],[-32.1002896772,-1.12096681644],[-31.9715117231,-0.837203498428],[-31.8800818691,-0.556468898942],[-31.8254440703,-0.277736442783],[-31.8072679929,0],[-31.8254440703,0.277736442783],[-31.8800818691,0.556468898942],[-31.9715117231,0.837203498428],[-32.1002896772,1.12096681644],[-32.2672058831,1.40881663388],[-32.4732966814,1.70185336507],[-32.7198607165,2.00123241238],[-33.0084795316,2.30817774075],[-33.3410431535,2.6239970035],[-33.7197810671,2.95009857994],[-34.1472983973,3.2880108612],[-34.6266155102,3.63940394304],[-35.1612056414,4.00611335337],[-35.7550182057,4.39016421046],[-36.4124633674,4.79379174563],[-37.138314422,5.21944970831],[-37.9374570742,5.66979094838],[-38.8143785616,6.14759362503],[-39.7722470564,6.65559159145],[-40.8113899535,7.19614916902],[-41.9269541846,7.77070164547],[-43.1055530065,8.37887071896],[-44.3208250346,9.01717376565],[-45.5281335086,9.67730351785],[-46.6592118851,10.344098238],[-47.6184628391,10.9935883798],[-48.28369535,11.5918896634],[-48.5147863138,12.09609478],[-48.1730192886,12.4583898795],[-47.1505372707,12.6339483844],[-45.4034911139,12.5915024719],[-42.9767802608,12.3233934346],[-40.0076792879,11.8508145072],[-36.7032455714,11.2213082803],[-33.2989492854,10.4991183806],[-30.0149182764,9.75243812881],[-27.0253656239,9.04256084558],[-24.4481119546,8.41816003875],[-22.3518922366,7.9152201722],[-20.7749251664,7.56145437968],[-19.7499695415,7.38421103185],[-19.3382230386,7.42324832601],[-19.694017676,7.75767986951],[-21.2841774283,8.59936587637],[-29.3860838838,12.1721144897],[-21.1308561166,8.96951626646],[-19.4112814895,8.44026540548],[-18.9232178765,8.42515941694],[-19.1867630853,8.74391169667],[-20.0368461313,9.3433347963],[-21.4020804312,10.2082687145],[-23.2399637988,11.3348877018],[-25.5038753899,12.7157632035],[-28.119767384,14.327737116],[-30.9699106495,16.1219150427],[-33.8877770139,18.0184506563],[-36.6694926246,19.9099100234],[-39.1031278208,21.6752176912],[-41.0086532394,23.201579673],[-42.2740152165,24.406914065],[-42.8728805757,25.2540566427],[-42.8583650351,25.7519037455],[-42.3384321905,25.945024615],[-41.4449788762,25.8976970902],[-40.3074271381,25.6786631194],[-39.0361388798,25.3503649976],[-37.7156606466,24.9634512129],[-36.4049851418,24.5554725336],[-35.1415494465,24.1521177904],[-33.9464564616,23.7695647099],[-32.8294395437,23.4170116495],[-31.7929154287,23.0989051482],[-30.8349807857,22.8166855306],[-29.9514622525,22.5700456868],[-29.1372224177,22.3577771159],[-28.3869307184,22.1783009509],[-27.6954768621,22.0299770253],[-27.0581618444,21.9112674292],[-26.4707610325,21.8208114794],[-25.9295205637,21.7574511416],[-25.4311237807,21.7202316307],[-24.9726478461,21.7083915808],[-24.5515204017,21.7213503826],[-24.1654803459,21.7586962359],[-23.8125438851,21.8201762946],[-23.4909757462,21.9056892714],[-23.1992650151,22.0152804732],[-22.9361050064,22.1491391425],[-22.7003766384,22.3075979943],[-22.4911348888,22.4911348888],[-22.3075979943,22.7003766384],[-22.1491391425,22.9361050064],[-22.0152804732,23.1992650151],[-21.9056892714,23.4909757462],[-21.8201762946,23.8125438851],[-21.7586962359,24.1654803459],[-21.7213503826,24.5515204017],[-21.7083915808,24.9726478461],[-21.7202316307,25.4311237807],[-21.7574511416,25.9295205637],[-21.8208114794,26.4707610325],[-21.9112674292,27.0581618444],[-22.0299770253,27.6954768621],[-22.1783009509,28.3869307184],[-22.3577771159,29.1372224177],[-22.5700456868,29.9514622525],[-22.8166855306,30.8349807857],[-23.0989051482,31.7929154287],[-23.4170116495,32.8294395437],[-23.7695647099,33.9464564616],[-24.1521177904,35.1415494465],[-24.5554725336,36.4049851418],[-24.9634512129,37.7156606466],[-25.3503649976,39.0361388798],[-25.6786631194,40.3074271381],[-25.8976970902,41.4449788762],[-25.945024615,42.3384321905],[-25.7519037455,42.8583650351],[-25.2540566427,42.8728805757],[-24.406914065,42.2740152165],[-23.201579673,41.0086532394],[-21.6752176912,39.1031278208],[-19.9099100234,36.6694926246],[-18.0184506563,33.8877770139],[-16.1219150427,30.9699106495],[-14.327737116,28.119767384],[-12.7157632035,25.5038753899],[-11.3348877018,23.2399637988],[-10.2082687145,21.4020804312],[-9.3433347963,20.0368461313],[-8.74391169667,19.1867630853],[-8.42515941694,18.9232178765],[-8.44026540548,19.4112814895],[-8.96951626646,21.1308561166],[-12.1721144897,29.3860838838],[-8.59936587637,21.2841774283],[-7.75767986951,19.694017676],[-7.42324832601,19.3382230386],[-7.38421103185,19.7499695415],[-7.56145437968,20.7749251664],[-7.9152201722,22.3518922366],[-8.41816003875,24.4481119546],[-9.04256084558,27.0253656239],[-9.75243812881,30.0149182764],[-10.4991183806,33.2989492854],[-11.2213082803,36.7032455714],[-11.8508145072,40.0076792879],[-12.3233934346,42.9767802608],[-12.5915024719,45.4034911139],[-12.6339483844,47.1505372707],[-12.4583898795,48.1730192886],[-12.09609478,48.5147863138],[-11.5918896634,48.28369535],[-10.9935883798,47.6184628391],[-10.344098238,46.6592118851],[-9.67730351785,45.5281335086],[-9.01717376565,44.3208250346],[-8.37887071896,43.1055530065],[-7.77070164547,41.9269541846],[-7.19614916902,40.8113899535],[-6.65559159145,39.7722470564],[-6.14759362503,38.8143785616],[-5.66979094838,37.9374570742],[-5.21944970831,37.138314422],[-4.79379174563,36.4124633674],[-4.39016421046,35.7550182057],[-4.00611335337,35.1612056414],[-3.63940394304,34.6266155102],[-3.2880108612,34.1472983973],[-2.95009857994,33.7197810671],[-2.6239970035,33.3410431535],[-2.30817774075,33.0084795316],[-2.00123241238,32.7198607165],[-1.70185336507,32.4732966814],[-1.40881663388,32.2672058831],[-1.12096681644,32.1002896772],[-0.837203498428,31.9715117231],[-0.556468898942,31.8800818691],[-0.277736442783,31.8254440703],[0,31.8072679929],[0.277736442783,31.8254440703],[0.556468898942,31.8800818691],[0.837203498428,31.9715117231],[1.12096681644,32.1002896772],[1.40881663388,32.2672058831],[1.70185336507,32.4732966814],[2.00123241238,32.7198607165],[2.30817774075,33.0084795316],[2.6239970035,33.3410431535],[2.95009857994,33.7197810671],[3.2880108612,34.1472983973],[3.63940394304,34.6266155102],[4.00611335337,35.1612056414],[4.39016421046,35.7550182057],[4.79379174563,36.4124633674],[5.21944970831,37.138314422],[5.66979094838,37.9374570742],[6.14759362503,38.8143785616],[6.65559159145,39.7722470564],[7.19614916902,40.8113899535],[7.77070164547,41.9269541846],[8.37887071896,43.1055530065],[9.01717376565,44.3208250346],[9.67730351785,45.5281335086],[10.344098238,46.6592118851],[10.9935883798,47.6184628391],[11.5918896634,48.28369535],[12.09609478,48.5147863138],[12.4583898795,48.1730192886],[12.6339483844,47.1505372707],[12.5915024719,45.4034911139],[12.3233934346,42.9767802608],[11.8508145072,40.0076792879],[11.2213082803,36.7032455714],[10.4991183806,33.2989492854],[9.75243812881,30.0149182764],[9.04256084558,27.0253656239],[8.41816003875,24.4481119546],[7.9152201722,22.3518922366],[7.56145437968,20.7749251664],[7.38421103185,19.7499695415],[7.42324832601,19.3382230386],[7.75767986951,19.694017676],[8.59936587637,21.2841774283],[12.1721144897,29.3860838838],[8.96951626646,21.1308561166],[8.44026540548,19.4112814895],[8.42515941694,18.9232178765],[8.74391169667,19.1867630853],[9.3433347963,20.0368461313],[10.2082687145,21.4020804312],[11.3348877018,23.2399637988],[12.7157632035,25.5038753899],[14.327737116,28.119767384],[16.1219150427,30.9699106495],[18.0184506563,33.8877770139],[19.9099100234,36.6694926246],[21.6752176912,39.1031278208],[23.201579673,41.0086532394],[24.406914065,42.2740152165],[25.2540566427,42.8728805757],[25.7519037455,42.8583650351],[25.945024615,42.3384321905],[25.8976970902,41.4449788762],[25.6786631194,40.3074271381],[25.3503649976,39.0361388798],[24.9634512129,37.7156606466],[24.5554725336,36.4049851418],[24.1521177904,35.1415494465],[23.7695647099,33.9464564616],[23.4170116495,32.8294395437],[23.0989051482,31.7929154287],[22.8166855306,30.8349807857],[22.5700456868,29.9514622525],[22.3577771159,29.1372224177],[22.1783009509,28.3869307184],[22.0299770253,27.6954768621],[21.9112674292,27.0581618444],[21.8208114794,26.4707610325],[21.7574511416,25.9295205637],[21.7202316307,25.4311237807],[21.7083915808,24.9726478461],[21.7213503826,24.5515204017],[21.7586962359,24.1654803459],[21.8201762946,23.8125438851],[21.9056892714,23.4909757462],[22.0152804732,23.1992650151],[22.1491391425,22.9361050064],[22.3075979943,22.7003766384],[22.4911348888,22.4911348888],[22.7003766384,22.3075979943],[22.9361050064,22.1491391425],[23.1992650151,22.0152804732],[23.4909757462,21.9056892714],[23.8125438851,21.8201762946],[24.1654803459,21.7586962359],[24.5515204017,21.7213503826],[24.9726478461,21.7083915808],[25.4311237807,21.7202316307],[25.9295205637,21.7574511416],[26.4707610325,21.8208114794],[27.0581618444,21.9112674292],[27.6954768621,22.0299770253],[28.3869307184,22.1783009509],[29.1372224177,22.3577771159],[29.9514622525,22.5700456868],[30.8349807857,22.8166855306],[31.7929154287,23.0989051482],[32.8294395437,23.4170116495],[33.9464564616,23.7695647099],[35.1415494465,24.1521177904],[36.4049851418,24.5554725336],[37.7156606466,24.9634512129],[39.0361388798,25.3503649976],[40.3074271381,25.6786631194],[41.4449788762,25.8976970902],[42.3384321905,25.945024615],[42.8583650351,25.7519037455],[42.8728805757,25.2540566427],[42.2740152165,24.406914065],[41.0086532394,23.201579673],[39.1031278208,21.6752176912],[36.6694926246,19.9099100234],[33.8877770139,18.0184506563],[30.9699106495,16.1219150427],[28.119767384,14.327737116],[25.5038753899,12.7157632035],[23.2399637988,11.3348877018],[21.4020804312,10.2082687145],[20.0368461313,9.3433347963],[19.1867630853,8.74391169667],[18.9232178765,8.42515941694],[19.4112814895,8.44026540548],[21.1308561166,8.96951626646],[29.3860838838,12.1721144897],[21.2841774283,8.59936587637],[19.694017676,7.75767986951],[19.3382230386,7.42324832601],[19.7499695415,7.38421103185],[20.7749251664,7.56145437968],[22.3518922366,7.9152201722],[24.4481119546,8.41816003875],[27.0253656239,9.04256084558],[30.0149182764,9.75243812881],[33.2989492854,10.4991183806],[36.7032455714,11.2213082803],[40.0076792879,11.8508145072],[42.9767802608,12.3233934346],[45.4034911139,12.5915024719],[47.1505372707,12.6339483844],[48.1730192886,12.4583898795],[48.5147863138,12.09609478],[48.28369535,11.5918896634],[47.6184628391,10.9935883798],[46.6592118851,10.344098238],[45.5281335086,9.67730351785],[44.3208250346,9.01717376565],[43.1055530065,8.37887071896],[41.9269541846,7.77070164547],[40.8113899535,7.19614916902],[39.7722470564,6.65559159145],[38.8143785616,6.14759362503],[37.9374570742,5.66979094838],[37.138314422,5.21944970831],[36.4124633674,4.79379174563],[35.7550182057,4.39016421046],[35.1612056414,4.00611335337],[34.6266155102,3.63940394304],[34.1472983973,3.2880108612],[33.7197810671,2.95009857994],[33.3410431535,2.6239970035],[33.0084795316,2.30817774075],[32.7198607165,2.00123241238],[32.4732966814,1.70185336507],[32.2672058831,1.40881663388],[32.1002896772,1.12096681644],[31.9715117231,0.837203498428],[31.8800818691,0.556468898942],[31.8254440703,0.277736442783]]);
    assert_approx(supershape(m1=7, n1=2, n2=8, n3=4), [[1,0],[1.00019517316,-0.00872857104113],[1.00078102433,-0.0174686977586],[1.0017585498,-0.0262319708163],[1.00312941419,-0.0350300510433],[1.00489595654,-0.0438747049874],[1.00706119869,-0.0527778410254],[1.00962885581,-0.0617515462009],[1.01260334878,-0.0708081239432],[1.01598981818,-0.0799601328075],[1.01979413959,-0.0892204263441],[1.0240229395,-0.0986021941772],[1.02868361163,-0.10811900433],[1.03378433254,-0.117784846781],[1.03933407606,-0.12761417816],[1.04534262515,-0.137621967435],[1.05182058018,-0.147823742296],[1.0587793619,-0.158235635844],[1.06623120726,-0.168874433019],[1.07418915584,-0.179757615985],[1.08266702407,-0.190903407467],[1.09167936393,-0.202330810682],[1.10124140209,-0.214059644174],[1.11136895468,-0.226110569338],[1.12207831186,-0.238505107894],[1.1333860853,-0.251265645826],[1.14530901022,-0.26441541947],[1.15786369227,-0.277978478424],[1.17106628758,-0.291979618679],[1.18493210244,-0.306444277958],[1.19947509676,-0.321398383517],[1.21470727308,-0.336868140672],[1.23063793043,-0.35287974809],[1.2472727598,-0.369459023352],[1.26461275593,-0.386630919653],[1.28265291846,-0.404418911741],[1.30138071516,-0.422844226653],[1.32077428165,-0.441924892758],[1.34080033604,-0.461674579605],[1.36141179612,-0.482101201868],[1.38254510094,-0.503205264274],[1.40411726123,-0.524977932173],[1.42602269563,-0.547398825996],[1.44812995451,-0.570433559133],[1.47027849121,-0.594031069729],[1.49227571289,-0.618120839079],[1.51389462641,-0.642610143307],[1.53487248259,-0.667381549418],[1.55491090311,-0.692290937163],[1.57367802873,-0.717166395447],[1.59081323057,-0.741808392109],[1.60593484806,-0.76599162963],[1.61865123028,-0.789468954531],[1.62857504526,-0.811977564899],[1.63534039155,-0.83324754808],[1.63862174143,-0.853012486907],[1.63815324577,-0.871021531282],[1.63374655172,-0.88705200202],[1.62530514368,-0.900921352515],[1.6128334029,-0.912497235111],[1.59643910952,-0.921704549628],[1.57632890991,-0.928528688623],[1.55279718806,-0.933014679643],[1.52620962151,-0.935262458929],[1.49698329573,-0.935418981821],[1.46556549563,-0.933668192529],[1.4324131807,-0.930219996144],[1.39797475806,-0.925299307277],[1.36267521949,-0.919136041237],[1.32690513459,-0.911956632875],[1.29101349317,-0.903977379849],[1.25530402436,-0.895399658682],[1.22003440231,-0.88640687891],[1.18541766257,-0.87716292795],[1.15162516709,-0.867811808863],[1.11879053251,-0.85847816949],[1.08701404382,-0.849268448251],[1.05636719081,-0.840272404763],[1.02689707061,-0.831564851513],[0.998630490424,-0.823207449243],[0.971577675677,-0.815250469321],[0.945735541708,-0.807734459729],[0.921090524428,-0.800691777214],[0.897620989299,-0.794147967228],[0.875299252405,-0.788122986837],[0.854093254551,-0.782632274663],[0.833967931608,-0.77768767758],[0.814886323466,-0.77329824687],[0.796810461115,-0.769470917947],[0.779702067617,-0.766211087893],[0.763523104482,-0.763523104482],[0.748236190758,-0.761410679404],[0.733804918133,-0.759877237135],[0.720194081714,-0.758926209655],[0.707369842875,-0.758561285914],[0.695299837765,-0.758786623746],[0.683953242625,-0.759607030817],[0.673300804976,-0.761028120197],[0.663314848035,-0.76305644522],[0.653969254213,-0.765699617546],[0.645239432346,-0.768966411581],[0.637102272286,-0.772866857811],[0.62953608962,-0.777412327008],[0.622520562564,-0.782615606762],[0.616036662471,-0.788490971254],[0.610066578882,-0.795054244721],[0.604593639556,-0.802322858557],[0.59960222555,-0.810315901451],[0.595077681001,-0.819054161405],[0.591006216921,-0.828560157837],[0.58737480797,-0.838858161213],[0.584171080778,-0.849974196821],[0.581383192055,-0.86193602829],[0.578999694296,-0.874773115235],[0.577009386474,-0.888516538024],[0.575401146653,-0.903198880958],[0.574163742953,-0.918854063171],[0.573285618773,-0.935517104201],[0.572754647627,-0.953223808466],[0.572557852412,-0.97201034972],[0.572681083409,-0.991912732999],[0.573108648924,-1.01296610763],[0.573822892216,-1.03520390063],[0.574803708461,-1.05865673543],[0.576027996016,-1.08335109689],[0.577469037497,-1.10930769991],[0.579095808431,-1.13653951732],[0.580872214881,-1.1650494232],[0.582756266925,-1.19482741278],[0.58469920279,-1.225847371],[0.586644589257,-1.25806338154],[0.588527438309,-1.29140559966],[0.590273398058,-1.3257757587],[0.591798097672,-1.3610424444],[0.593006750426,-1.39703635594],[0.593794143966,-1.43354587562],[0.594045168693,-1.47031338737],[0.593636048218,-1.5070329046],[0.592436432393,-1.54334967155],[0.590312484623,-1.57886245951],[0.587131032396,-1.61312925377],[0.582764747729,-1.64567688039],[0.577098183987,-1.67601482342],[0.570034329323,-1.70365302829],[0.561501169897,-1.7281229067],[0.551457624266,-1.74900013484],[0.539898154885,-1.76592729362],[0.526855416953,-1.77863407952],[0.512400481582,-1.7869528405],[0.496640451325,-1.79082761322],[0.479713622154,-1.79031561096],[0.461782668252,-1.78558109033],[0.443026566032,-1.77688250409],[0.423632085046,-1.76455462645],[0.403785643857,-1.74898777475],[0.383666181409,-1.73060630705],[0.36343947542,-1.70984829903],[0.343254102074,-1.68714781326],[0.323239020449,-1.66292060077],[0.303502608508,-1.63755353666],[0.284132884147,-1.61139766022],[0.265198609405,-1.58476439958],[0.246750984994,-1.55792440525],[0.228825679411,-1.53110837228],[0.211444987706,-1.50450926347],[0.194619968287,-1.47828542453],[0.178352454935,-1.45256417927],[0.162636881916,-1.42744559265],[0.14746189153,-1.40300617913],[0.132811716116,-1.37930240876],[0.118667341773,-1.35637392309],[0.105007470326,-1.33424641679],[0.0918093009639,-1.31293417235],[0.079049154786,-1.29244225626],[0.0667029652744,-1.27276839788],[0.0547466562535,-1.25390457939],[0.0431564267959,-1.23583836851],[0.0319089601471,-1.21855402579],[0.0209815713408,-1.20203341707],[0.0103523059058,-1.18625675947],[0,-1.17120322617],[-0.0100956895061,-1.15685143262],[-0.0199542780464,-1.14317982365],[-0.0295944638537,-1.13016697829],[-0.0390341514226,-1.11779184682],[-0.0482904843594,-1.10603393199],[-0.0573798847909,-1.09487342482],[-0.0663180971294,-1.08429130352],[-0.0751202344883,-1.07426940254],[-0.083800826454,-1.06479045798],[-0.0923738672398,-1.05583813396],[-0.100852863514,-1.04739703426],[-0.109250881394,-1.03945270253],[-0.117580592275,-1.03199161375],[-0.125854317275,-1.02500115934],[-0.134084070198,-1.01846962766],[-0.142281598985,-1.01238618147],[-0.150458425682,-1.00674083361],[-0.158625885026,-1.00152442185],[-0.166795161748,-0.996728583735],[-0.174977326756,-0.992345732077],[-0.183183372357,-0.988369031602],[-0.191424246709,-0.984792377159],[-0.199710887689,-0.981610373805],[-0.208054256382,-0.978818318978],[-0.216465370386,-0.97641218695],[-0.224955337149,-0.974388615652],[-0.233535387516,-0.972744895929],[-0.242216909709,-0.971478963239],[-0.251011483899,-0.970589391764],[-0.259930917575,-0.970075390849],[-0.268987281843,-0.969936803664],[-0.278192948825,-0.970174107906],[-0.287560630268,-0.97078841835],[-0.29710341746,-0.971781490959],[-0.306834822501,-0.973155728228],[-0.31676882096,-0.97491418536],[-0.326919895858,-0.977060576773],[-0.337303082869,-0.979599282343],[-0.347934016547,-0.982535352666],[-0.358828977261,-0.985874512471],[-0.370004938413,-0.989623161138],[-0.381479613327,-0.993788369082],[-0.393271501017,-0.998377868481],[-0.405399929774,-1.00340003656],[-0.417885097231,-1.00886386925],[-0.430748105166,-1.01477894263],[-0.44401098687,-1.02115535904],[-0.457696724337,-1.02800367416],[-0.471829251861,-1.03533480051],[-0.486433441802,-1.04315988231],[-0.501535067296,-1.05149013506],[-0.517160735507,-1.06033664272],[-0.533337783579,-1.06971010355],[-0.550094127821,-1.07962051428],[-0.567458054652,-1.09007678079],[-0.585457939643,-1.10108624143],[-0.604121878377,-1.11265408721],[-0.623477210088,-1.12478266132],[-0.643549911948,-1.13747061852],[-0.66436383883,-1.15071192356],[-0.685939780411,-1.16449466728],[-0.708294305129,-1.17879967949],[-0.731438359227,-1.19359892054],[-0.755375589777,-1.20885363871],[-0.780100364279,-1.22451229051],[-0.805595467669,-1.24050823567],[-0.83182947214,-1.2567572416],[-0.858753798386,-1.27315486305],[-0.886299520912,-1.28957380504],[-0.914374017105,-1.30586142995],[-0.942857621036,-1.32183763375],[-0.971600517789,-1.3372933866],[-1.00042019865,-1.35199030383],[-1.02909988309,-1.36566167078],[-1.0573883855,-1.37801537293],[-1.0850019415,-1.38873915593],[-1.11162848437,-1.39750853724],[-1.13693474894,-1.40399748864],[-1.16057635812,-1.40789170306],[-1.18221071411,-1.40890386574],[-1.20151209783,-1.40678991842],[-1.21818793724,-1.40136491706],[-1.23199482864,-1.39251683884],[-1.24275268616,-1.38021668609],[-1.25035543709,-1.36452351786],[-1.25477700524,-1.34558359848],[-1.25607189213,-1.32362359585],[-1.25437036725,-1.29893854001],[-1.24986896766,-1.27187590174],[-1.24281754181,-1.24281754181],[-1.2335043604,-1.212161359],[-1.22224083334,-1.18030425287],[-1.20934716014,-1.14762760383],[-1.19513988395,-1.11448597183],[-1.17992191314,-1.08119923192],[-1.16397519694,-1.04804797479],[-1.14755594421,-1.01527173633],[-1.13089207635,-0.983069483869],[-1.11418250205,-0.951601755085],[-1.09759777507,-0.920993888242],[-1.08128172222,-0.891339866943],[-1.0653536858,-0.862706404464],[-1.04991109386,-0.835136993363],[-1.03503214162,-0.808655735217],[-1.0207784301,-0.783270838159],[-1.00719746045,-0.758977725579],[-0.994324923533,-0.735761739345],[-0.982186754793,-0.713600447801],[-0.970800946481,-0.692465585433],[-0.96017912326,-0.672324660138],[-0.95032789653,-0.653142267713],[-0.941250017539,-0.634881153308],[-0.93294535152,-0.617503057556],[-0.925411695294,-0.600969381758],[-0.918645459826,-0.585241702681],[-0.912642237564,-0.570282163512],[-0.907397272447,-0.556053763711],[-0.902905848327,-0.542520566949],[-0.899163609551,-0.529647843153],[-0.896166825497,-0.517402157939],[-0.893912609103,-0.505751420309],[-0.892399097874,-0.49466489746],[-0.89162560444,-0.484113203874],[-0.891592742532,-0.474068270405],[-0.892302533195,-0.464503297905],[-0.893758495109,-0.45539269896],[-0.895965722075,-0.446712030473],[-0.898930950017,-0.438437919194],[-0.902662615161,-0.430547981713],[-0.907170904456,-0.423020740003],[-0.912467798696,-0.415835533194],[-0.918567108228,-0.408972425964],[-0.925484500488,-0.402412113647],[-0.933237517977,-0.396135823923],[-0.941845584544,-0.390125214779],[-0.951329997017,-0.384362268218],[-0.961713898268,-0.378829179072],[-0.973022226692,-0.373508238117],[-0.985281635748,-0.368381708584],[-0.998520375674,-0.363431695054],[-1.01276812765,-0.358640003669],[-1.02805577856,-0.35398799256],[-1.0444151219,-0.349456411405],[-1.06187846767,-0.345025229153],[-1.08047814056,-0.340673449129],[-1.10024584217,-0.336378911098],[-1.12121184898,-0.33211808038],[-1.14340401359,-0.327865824955],[-1.16684653232,-0.323595182563],[-1.19155843905,-0.319277121479],[-1.21755178197,-0.314880300681],[-1.24482943955,-0.310370838044],[-1.27338253466,-0.305712098756],[-1.30318741304,-0.300864520746],[-1.33420216748,-0.295785499417],[-1.36636271369,-0.290429360417],[-1.39957846187,-0.284747456291],[-1.4337276823,-0.278688430105],[-1.46865273669,-0.272198695555],[-1.5041554402,-0.265223187287],[-1.53999293024,-0.257706434913],[-1.57587453976,-0.249594007002],[-1.61146028981,-0.240834353947],[-1.6463617054,-0.231381048299],[-1.68014568705,-0.221195376011],[-1.71234209877,-0.210249172714],[-1.74245552511,-0.198527730193],[-1.76998128115,-0.186032528989],[-1.79442523791,-0.17278349822],[-1.81532640203,-0.158820480777],[-1.83228056355,-0.144203607733],[-1.84496283835,-0.12901236944],[-1.85314672619,-0.113343309297],[-1.85671748567,-0.0973064401801],[-1.85567821356,-0.0810206605386],[-1.85014792024,-0.0646085890484],[-1.8403519368,-0.0481913114767],[-1.82660595241,-0.0318835254976],[-1.80929567112,-0.0157894841163],[-1.788854382,0],[-1.76574065129,0.0154093852166],[-1.74041795236,0.0303791083606],[-1.71333748273,0.0448653209444],[-1.684924815,0.0588388710756],[-1.65557049658,0.0722837689322],[-1.62562431445,0.0851953602689],[-1.59539269171,0.0975783971936],[-1.56513856936,0.109445150405],[-1.53508311778,0.120813661487],[-1.50540868122,0.131706193581],[-1.47626245461,0.142147906644],[-1.4477604993,0.152165760127],[-1.41999180796,0.161787630423],[-1.39302221876,0.171041621458],[-1.36689805277,0.179955542595],[-1.34164940462,0.188556527203],[-1.31729305748,0.19687076651],[-1.29383502189,0.204923335815],[-1.27127271587,0.212738093127],[-1.24959681467,0.220337633434],[-1.22879280385,0.227743284684],[-1.20884227079,0.234975134295],[-1.1897239693,0.242052077233],[-1.17141468962,0.248991878715],[-1.15388996324,0.255811246128],[-1.13712462872,0.262525906116],[-1.12109328127,0.269150683781],[-1.10577062611,0.275699581811],[-1.09113175211,0.282185857944],[-1.07715234033,0.288622099717],[-1.06380881938,0.295020295801],[-1.05107847759,0.30139190352],[-1.03893954059,0.307747912371],[-1.02737122101,0.314098903509],[-1.01635374635,0.320455105296],[-1.00586836951,0.326826445071],[-0.995897366039,0.333222597381],[-0.986424021257,0.339653028931],[-0.977432609846,0.346127040544],[-0.968908370071,0.352653806437],[-0.960837474336,0.359242411129],[-0.953206997474,0.365901884274],[-0.9460048839,0.372641233751],[-0.939219914518,0.379469477292],[-0.932841674097,0.386395672958],[-0.926860519664,0.393428948736],[-0.921267550353,0.400578531544],[-0.916054579033,0.407853775894],[-0.911214105938,0.415264192488],[-0.906739294482,0.422819476967],[-0.902623949342,0.430529539063],[-0.898862496855,0.438404532356],[-0.895449967744,0.446454884849],[-0.892381982103,0.454691330552],[-0.88965473657,0.463124942226],[-0.887264993572,0.471767165465],[-0.885210072453,0.480629854216],[-0.883487842309,0.48972530784],[-0.882096716258,0.499066309787],[-0.881035646865,0.508666167883],[-0.880304122354,0.518538756207],[-0.879902163198,0.528698558463],[-0.879830318582,0.539160712656],[-0.880089662167,0.549941056821],[-0.880681786461,0.561056175389],[-0.881608794973,0.572523445685],[-0.882873291204,0.584361083813],[-0.884478363322,0.596588189024],[-0.886427563174,0.609224785349],[-0.888724878027,0.622291858989],[-0.891374693143,0.635811389517],[-0.894381742924,0.649806372506],[-0.897751047959,0.664300830556],[-0.901487834796,0.679319809029],[-0.905597434692,0.694889351884],[-0.910085156921,0.711036451999],[-0.914956131422,0.727788969109],[-0.920215114699,0.745175506988],[-0.925866251831,0.763225239785],[-0.931912786371,0.781967675333],[-0.938356708637,0.801432340922],[-0.945198331629,0.821648374291],[-0.952435782493,0.842643999638],[-0.960064396258,0.864445865177],[-0.968075997571,0.887078215393],[-0.976458055709,0.910561867929],[-0.985192698395,0.934912962163],[-0.994255571559,0.960141444744],[-1.00361453554,0.986249257294],[-1.01322819429,1.01322819429],[-1.02304426394,1.04105740632],[-1.03299780123,1.06970053725],[-1.04300933324,1.09910250585],[-1.05298295712,1.1291859754],[-1.06280451426,1.15984760139],[-1.07233998721,1.19095420995],[-1.08143431712,1.22233913786],[-1.08991089227,1.25379905715],[-1.09757200558,1.28509170651],[-1.10420061022,1.31593504418],[-1.10956370259,1.34600840346],[-1.11341761212,1.37495624324],[-1.11551536117,1.40239501109],[-1.11561606399,1.42792344584],[-1.11349606448,1.45113632379],[-1.10896119306,1.47164120862],[-1.10185920091,1.48907724757],[-1.09209117849,1.50313455357],[-1.07962065886,1.51357234138],[-1.06447920953,1.52023386131],[-1.04676764605,1.52305637592],[-1.02665252018,1.52207495464],[-1.00435814786,1.51741963677],[-0.980155021956,1.50930637742],[-0.954345879367,1.49802296242],[-0.927250883113,1.48391160533],[-0.899193322693,1.46735014069],[-0.870486974871,1.44873361193],[-0.841425886442,1.42845769506],[-0.812276932107,1.40690491623],[-0.783275140128,1.38443412325],[-0.754621510837,1.36137324271],[-0.726482892651,1.33801504089],[-0.698993416855,1.31461541818],[-0.672257003249,1.29139368797],[-0.646350506804,1.26853429489],[-0.621327156886,1.24618948403],[-0.59722002709,1.22448251581],[-0.574045353793,1.20351111208],[-0.551805588939,1.18335090425],[-0.530492125401,1.16405872815],[-0.510087672036,1.14567566931],[-0.490568282149,1.12822980761],[-0.471905056003,1.11173864263],[-0.454065547496,1.09621120297],[-0.437014909438,1.0816498571],[-0.420716812624,1.06805185104],[-0.405134172367,1.05541060217],[-0.390229713439,1.04371677913],[-0.375966400949,1.03295919708],[-0.362307761129,1.02312555521],[-0.34921811256,1.01420304118],[-0.336662725103,1.00617882403],[-0.324607920971,0.99904045441],[-0.313021129835,0.992776188415],[-0.301870907659,0.987375248762],[-0.29112692718,0.982828034951],[-0.280759946339,0.979126292113],[-0.270741759743,0.976263246591],[-0.261045137137,0.974233714864],[-0.251643752018,0.973034191176],[-0.242512102818,0.972662918134],[-0.233625428507,0.973119943649],[-0.224959619995,0.974407166695],[-0.216491128366,0.976528373689],[-0.20819687069,0.979489266547],[-0.200054133902,0.983297482846],[-0.192040477133,0.987962607864],[-0.184133632701,0.993496177607],[-0.176311405961,0.999911671219],[-0.168551574153,1.00722449039],[-0.160831784425,1.01545192252],[-0.153129451303,1.02461308336],[-0.145421653975,1.03472883367],[-0.137685033994,1.04582166322],[-0.129895694255,1.05791553352],[-0.122029100482,1.07103566919],[-0.114059986943,1.08520828542],[-0.105962268744,1.10046023642],[-0.0977089638307,1.11681856703],[-0.0892721288411,1.13430994629],[-0.0806228141582,1.15295995795],[-0.0717310450376,1.17279221951],[-0.0625658374775,1.19382729689],[-0.0530952596407,1.21608137854],[-0.043286552099,1.23956466965],[-0.0331063229204,1.26427946532],[-0.0225208365644,1.29021786267],[-0.01149641849,1.31735907609],[0,1.34566633035],[0.0120001703725,1.37508332431],[0.0245336224199,1.4055302871],[0.0376265426254,1.43689969154],[0.0513004175008,1.46905174907],[0.0655704358244,1.50180988903],[0.0804436626124,1.53495652197],[0.0959170251376,1.5682294987],[0.111975188063,1.60131979352],[0.128588439961,1.63387104484],[0.14571076388,1.66548165222],[0.163278313627,1.69571012159],[0.181208554772,1.72408423232],[0.199400341614,1.75011433727],[0.217735174206,1.77331068829],[0.236079801695,1.79320412465],[0.254290206952,1.80936883925],[0.272216833096,1.82144536094],[0.289710721026,1.82916150369],[0.306630056733,1.83234896609],[0.322846519459,1.83095359634],[0.338250808935,1.82503804884],[0.352756822464,1.81477652767],[0.366304134218,1.80044234081],[0.378858660556,1.78238986169],[0.390411626539,1.76103304376],[0.400977134938,1.73682278613],[0.410588751572,1.71022523253],[0.419295553175,1.68170261019],[0.427158048022,1.65169761788],[0.434244298388,1.62062178448],[0.440626472807,1.58884773182],[0.446377956374,1.55670493247],[0.451571062913,1.52447835967],[0.456275330155,1.49240935839],[0.46055633877,1.46069809024],[0.464474975104,1.42950698431],[0.46808705111,1.39896473091],[0.471443198457,1.36917046517],[0.474588963001,1.34019788798],[0.477565037556,1.31209915701],[0.480407583188,1.28490844784],[0.483148600574,1.25864513598],[0.485816322948,1.23331658593],[0.488435610295,1.20892055778],[0.491028330979,1.18544725616],[0.493613721995,1.16288105476],[0.49620872272,1.14120193282],[0.498828279765,1.12038666022],[0.501485622362,1.10040976633],[0.504192508983,1.08124432478],[0.506959446695,1.06286258297],[0.50979588519,1.04523646183],[0.512710387684,1.0283379479],[0.515710780918,1.01213939643],[0.518804286479,0.996613761722],[0.52199763554,0.981734768008],[0.525297168987,0.967477032356],[0.528708924732,0.953816148855],[0.532238713852,0.940728741946],[0.535892187032,0.928192495319],[0.539674892645,0.916186161669],[0.543592327629,0.904689557636],[0.54764998225,0.893683547491],[0.551853379661,0.883150018439],[0.556208111104,0.873071849875],[0.560719867508,0.86343287849],[0.565394468134,0.854217860732],[0.570237886873,0.845412433845],[0.57525627672,0.837003076429],[0.580455992917,0.828977069286],[0.58584361518,0.821322457131],[0.591425969415,0.814028011599],[0.597210149272,0.807083195896],[0.60320353786,0.8004781313],[0.609413829903,0.79420356569],[0.615849054628,0.788250844165],[0.622517599597,0.782611881803],[0.629428235718,0.777279138531],[0.636590143613,0.772245596052],[0.64401294152,0.767504736734],[0.651706714856,0.763050524322],[0.659682047563,0.758877386327],[0.667950055296,0.754980197871],[0.676522420507,0.751354266778],[0.685411429405,0.74799531964],[0.694630010718,0.744899488538],[0.704191776145,0.742063298088],[0.714111062268,0.739483652391],[0.724402973617,0.737157821444],[0.735083426459,0.735083426459],[0.746169192726,0.733258423513],[0.757677943328,0.731681084791],[0.769628289879,0.730349976636],[0.782039823579,0.729263933448],[0.7949331497,0.72842202633],[0.808329915697,0.727823525221],[0.822252830504,0.727467853],[0.836725671988,0.72735452985],[0.851773278817,0.727483105831],[0.86742152215,0.727853079311],[0.88369725155,0.728463798499],[0.900628208251,0.729314342887],[0.918242897504,0.730403380899],[0.936570409951,0.731728999506],[0.955640179993,0.733288500906],[0.975481666747,0.735078160778],[0.996123940522,0.73709294189],[1.01759515474,0.73932615621],[1.03992187993,0.741769068106],[1.06312827296,0.744410430808],[1.08723505121,0.74723594822],[1.11225823821,0.750227654602],[1.13820764484,0.75336320576],[1.16508504931,0.756615077741],[1.19288204069,0.759949672772],[1.22157749596,0.763326338211],[1.25113467173,0.766696312905],[1.28149791021,0.770001627609],[1.31258898864,0.773174002366],[1.34430318337,0.776133804792],[1.37650517819,0.778789158855],[1.40902502187,0.781035323344],[1.44165443194,0.78275449073],[1.47414384597,0.783816186528],[1.50620072771,0.784078470363],[1.53748972645,0.783390143963],[1.56763533728,0.7815941473],[1.59622768543,0.778532260957],[1.62283192465,0.774051121739],[1.64700146895,0.768009397965],[1.66829486285,0.760285770973],[1.6862955621,0.750787156155],[1.70063331758,0.739456411754],[1.71100534046,0.726278677427],[1.71719511123,0.711285504312],[1.71908669707,0.694556110099],[1.71667282025,0.676215407099],[1.71005564131,0.656428858608],[1.69944015616,0.635394638091],[1.6851210624,0.613333907849],[1.66746472775,0.590480229136],[1.64688834709,0.567069133908],[1.62383845109,0.543328745362],[1.5987706712,0.519472080833],[1.57213218238,0.495691373063],[1.54434767007,0.47215446558],[1.51580911996,0.449003117119],[1.486869288,0.426352907562],[1.45783840825,0.404294371881],[1.42898353652,0.382894984607],[1.40052988308,0.362201655181],[1.37266352397,0.342243455008],[1.34553496406,0.323034364471],[1.31926312913,0.30457589224],[1.29393946908,0.286859474078],[1.26963194909,0.269868601685],[1.24638878411,0.253580664103],[1.22424183371,0.237968506097],[1.20320961984,0.223001721794],[1.18329996149,0.208647709482],[1.16451224147,0.194872516794],[1.14683933266,0.181641505845],[1.13026921778,0.168919866391],[1.11478633889,0.156673002583],[1.10037271259,0.14486681589],[1.08700884464,0.133467903686],[1.07467447485,0.122443690019],[1.06334917981,0.111762502365],[1.05301285719,0.101393605758],[1.04364611248,0.0913072035752],[1.03523056579,0.0814744124843],[1.02774909333,0.0718672175744],[1.02118601635,0.0624584124209],[1.01552724747,0.0532215278414],[1.01076040318,0.0441307522573],[1.00687488918,0.0351608459123],[1.00386196436,0.0262870506649],[1.00171478749,0.0174849966552],[1.00042845044,0.00873060682107]]);
}
test_supershape();


module test_reuleaux_polygon() {
    $fn=36;
    assert_approx(reuleaux_polygon(n=3, r=50),[[50,0],[45.5443467787,-6.93313174371],[40.449833029,-13.4113329645],[34.7625954562,-19.375936069],[28.5341385645,-24.7729246878],[21.8208682239,-29.5534228563],[14.6835808504,-33.6741376427],[7.18691282348,-37.0977512159],[-0.601244870218,-39.7932588011],[-8.61036146861,-41.7362494642],[-16.7679051715,-42.909127181],[-25,-43.3012701892],[-28.7764416072,-35.9759954373],[-31.8394715604,-28.3249164997],[-34.161350586,-20.417322732],[-35.7210513879,-12.324826528],[-36.5044490743,-4.12071478641],[-36.5044490743,4.12071478641],[-35.7210513879,12.324826528],[-34.161350586,20.417322732],[-31.8394715604,28.3249164997],[-28.7764416072,35.9759954373],[-25,43.3012701892],[-16.7679051715,42.909127181],[-8.61036146861,41.7362494642],[-0.601244870218,39.7932588011],[7.18691282348,37.0977512159],[14.6835808504,33.6741376427],[21.8208682239,29.5534228563],[28.5341385645,24.7729246878],[34.7625954562,19.375936069],[40.449833029,13.4113329645],[45.5443467787,6.93313174371]]);
    assert_approx(reuleaux_polygon(n=3, d=100),[[50,0],[45.5443467787,-6.93313174371],[40.449833029,-13.4113329645],[34.7625954562,-19.375936069],[28.5341385645,-24.7729246878],[21.8208682239,-29.5534228563],[14.6835808504,-33.6741376427],[7.18691282348,-37.0977512159],[-0.601244870218,-39.7932588011],[-8.61036146861,-41.7362494642],[-16.7679051715,-42.909127181],[-25,-43.3012701892],[-28.7764416072,-35.9759954373],[-31.8394715604,-28.3249164997],[-34.161350586,-20.417322732],[-35.7210513879,-12.324826528],[-36.5044490743,-4.12071478641],[-36.5044490743,4.12071478641],[-35.7210513879,12.324826528],[-34.161350586,20.417322732],[-31.8394715604,28.3249164997],[-28.7764416072,35.9759954373],[-25,43.3012701892],[-16.7679051715,42.909127181],[-8.61036146861,41.7362494642],[-0.601244870218,39.7932588011],[7.18691282348,37.0977512159],[14.6835808504,33.6741376427],[21.8208682239,29.5534228563],[28.5341385645,24.7729246878],[34.7625954562,19.375936069],[40.449833029,13.4113329645],[45.5443467787,6.93313174371]]);
    assert_approx(reuleaux_polygon(n=5, d=100),[[50,0],[47.0014382812,-7.98963912753],[43.2987621605,-15.6783253167],[38.921783409,-23.0041537871],[33.9057428858,-29.9081412755],[28.2910268,-36.3347009341],[22.122841544,-42.2320898832],[15.4508497187,-47.5528258148],[6.92564483387,-47.1699584219],[-1.53092011402,-46.0244388665],[-9.85075783633,-44.1254901984],[-17.9668818917,-41.4884016357],[-25.8139460215,-38.134405465],[-33.3287702792,-34.0905060913],[-40.4508497187,-29.3892626146],[-42.7211543799,-21.1629984251],[-44.2449228251,-12.7663422159],[-45.0098865668,-4.26689892573],[-45.0098865668,4.26689892573],[-44.2449228251,12.7663422159],[-42.7211543799,21.1629984251],[-40.4508497187,29.3892626146],[-33.3287702792,34.0905060913],[-25.8139460215,38.134405465],[-17.9668818917,41.4884016357],[-9.85075783633,44.1254901984],[-1.53092011402,46.0244388665],[6.92564483387,47.1699584219],[15.4508497187,47.5528258148],[22.122841544,42.2320898832],[28.2910268,36.3347009341],[33.9057428858,29.9081412755],[38.921783409,23.0041537871],[43.2987621605,15.6783253167],[47.0014382812,7.98963912753]]);
}
test_reuleaux_polygon();



// vim: expandtab tabstop=4 shiftwidth=4 softtabstop=4 nowrap
